<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Http\Requests\StoreRole;
use Yajra\Datatables\Datatables;
use App\Http\Requests\UpdateRole;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RoleController extends Controller
{
    public function index(Request $request)
    {
        if(!auth()->user()->can('view_role')) {
            abort(403, 'Unauthorized Action');
        }

        if ($request->ajax()) {
            $role = Role::query();

            return Datatables::of($role)
                ->addColumn('permissions', function($each) {
                    $output = '';
                    foreach($each->permissions as $permission){
                        $output .= '<span class="badge badge-pill badge-primary m-1">'.$permission->name.'</span>';
                    }
                    return $output;
                })
                ->editColumn('updated_at', function($each) {
                    return Carbon::parse($each->updated_at)->format('Y-m-d H:i:s');
                })
                ->addColumn('action', function($each) {
                    $edit_icon = '<a href="' . route('role.edit', $each->id) . '" class="text-warning"><i class="far fa-edit"></i></a>';
                    $delete_icon = '<a href="#" class="text-danger delete-btn" data-id="'. $each->id .'"><i class="fas fa-trash-alt"></i></a>';

                    return '<div class="action-icon">' . $edit_icon . $delete_icon.'</div>';
                })
                ->addColumn('plus-icon', function($each) {
                    return null;
                })
                ->rawColumns(['permissions', 'action'])
                ->make(true);

        }
        return view('dashboards.admins.role.index');
    }


    public function create()
    {
        if(!auth()->user()->can('create_role')) {
            abort(403, 'Unauthorized Action');
        }

        $permissions = Permission::all();
        return view('dashboards.admins.role.create', compact('permissions'));
    }

    public function store(StoreRole $request)
    {
        if(!auth()->user()->can('create_role')) {
            abort(403, 'Unauthorized Action');
        }

        $role = new Role();
        $role->name = $request->name;
        $role->save();

        $role->givePermissionTo($request->permissions);

        return redirect()->route('role.index')->with('create', 'Role is successfully Create!');
    }

    public function edit($id)
    {
        if(!auth()->user()->can('edit_role')) {
            abort(403, 'Unauthorized Action');
        }

        $role = Role::findOrFail($id);
        $old_permissions = $role->permissions->pluck('id')->toarray();
        $permissions = Permission::all();
        return view('dashboards.admins.role.edit', compact('role', 'old_permissions', 'permissions'));
    }

    public function update($id, UpdateRole $request)
    {
        if(!auth()->user()->can('edit_role')) {
            abort(403, 'Unauthorized Action');
        }

        $role = Role::findOrFail($id);

        $role->name = $request->name;
        $role->update();

        $old_permissions = $role->permissions->pluck('name')->toarray();
        $role->revokePermissionTo($old_permissions);

        $role->givePermissionTo($request->permissions);

        return redirect()->route('role.index')->with('update', 'Role is successfully Update!');
    }


    public function destroy($id)
    {
        if(!auth()->user()->can('delete_role')) {
            abort(403, 'Unauthorized Action');
        }

        $role = Role::findOrFail($id);
        $role->delete();

        return 'success';
    }
}
